<?php declare(strict_types=1);
defined('MW_PATH') or exit('No direct script access allowed');

/**
 * This file is part of the MailWizz EMA application.
 *
 * @package MailWizz EMA
 * @author MailWizz Development Team <support@mailwizz.com>
 * @link https://www.mailwizz.com/
 * @copyright MailWizz EMA (https://www.mailwizz.com)
 * @license https://www.mailwizz.com/license/
 * @since 1.3.5.3
 */

/** @var Controller $controller */
$controller = controller();

/** @var string $pageHeading */
$pageHeading = (string)$controller->getData('pageHeading');

/** @var DeliveryServerDynWebApi $server */
$server = $controller->getData('server');

/** @var bool $canSelectTrackingDomains */
$canSelectTrackingDomains = (bool)$controller->getData('canSelectTrackingDomains');

/**
 * This hook gives a chance to prepend content or to replace the default view content with a custom content.
 * Please note that from inside the action callback you can access all the controller view
 * variables via {@CAttributeCollection $collection->controller->getData()}
 * In case the content is replaced, make sure to set {@CAttributeCollection $collection->add('renderContent', false)}
 * in order to stop rendering the default content.
 * @since 1.3.3.1
 */
hooks()->doAction('before_view_file_content', $viewCollection = new CAttributeCollection([
    'controller'    => $controller,
    'renderContent' => true,
]));

// and render if allowed
if (!empty($viewCollection) && $viewCollection->itemAt('renderContent')) {
    $controller->renderPartial('_confirm-form');
    /**
     * This hook gives a chance to prepend content before the active form or to replace the default active form entirely.
     * Please note that from inside the action callback you can access all the controller view variables
     * via {@CAttributeCollection $collection->controller->getData()}
     * In case the form is replaced, make sure to set {@CAttributeCollection $collection->add('renderForm', false)}
     * in order to stop rendering the default content.
     * @since 1.3.3.1
     */
    hooks()->doAction('before_active_form', $collection = new CAttributeCollection([
        'controller'    => $controller,
        'renderForm'    => true,
    ]));

    // and render if allowed
    if (!empty($collection) && $collection->itemAt('renderForm')) {
        /** @var CActiveForm $form */
        $form = $controller->beginWidget('CActiveForm'); ?>
        <div class="box box-primary borderless">
            <div class="box-header">
                <div class="pull-left">
                    <?php BoxHeaderContent::make(BoxHeaderContent::LEFT)
                        ->add('<h3 class="box-title">' . IconHelper::make('glyphicon-send') . html_encode((string)$pageHeading) . '</h3>')
                        ->render(); ?>
                </div>
                <div class="pull-right">
                    <?php BoxHeaderContent::make(BoxHeaderContent::RIGHT)
                        ->add(CHtml::link(IconHelper::make('cancel') . t('app', 'Cancel'), ['delivery_servers/index'], ['class' => 'btn btn-primary btn-flat', 'title' => t('app', 'Cancel')]))
                        ->addIf(CHtml::link(IconHelper::make('info'), '#page-info', ['class' => 'btn btn-primary btn-flat', 'title' => t('app', 'Info'), 'data-toggle' => 'modal']), !$server->getIsNewRecord())
                        ->render(); ?>
                </div>
                <div class="clearfix"><!-- --></div>
            </div>
            <div class="box-body">
                <?php
                /**
                 * This hook gives a chance to prepend content before the active form fields.
                 * Please note that from inside the action callback you can access all the controller view variables
                 * via {@CAttributeCollection $collection->controller->getData()}
                 * @since 1.3.3.1
                 */
                hooks()->doAction('before_active_form_fields', new CAttributeCollection([
                    'controller'    => $controller,
                    'form'          => $form,
                ])); ?>
                <div class="row">
                    <?php
                    $index = 0;
        $formFields = $server->getFormFieldsDefinition([
                        'tracking_domain_id' => [
                            'visible' => !empty($canSelectTrackingDomains),
                        ],
                    ]);
        foreach ($formFields as $fieldName => $fieldProps) {
            $index++; ?>
                    <div class="col-lg-3">
                        <div class="form-group">
                            <?php echo $form->labelEx($server, $fieldName); ?>
                            <?php echo $fieldProps['fieldHtml']; ?>
                            <?php echo $form->error($server, $fieldName); ?>
                        </div>
                    </div>
                    <?php if ($index % 4 === 0) { ?></div><div class="row"><?php } ?>
                    <?php
        } ?>
                </div>
                <?php
                /**
                 * This hook gives a chance to append content after the active form fields.
                 * Please note that from inside the action callback you can access all the controller view variables
                 * via {@CAttributeCollection $collection->controller->getData()}
                 * @since 1.3.3.1
                 */
                hooks()->doAction('after_active_form_fields', new CAttributeCollection([
                    'controller'    => $controller,
                    'form'          => $form,
                ])); ?>
                <div class="row">
                    <div class="col-lg-12">
                        <?php $controller->renderPartial('_policies', compact('form')); ?>
                    </div>
                </div>
                <div class="row">
                    <div class="col-lg-12">
                        <?php $controller->renderPartial('_additional-headers'); ?>
                    </div>
                </div>
            </div>
            <div class="box-footer">
                <div class="pull-right">
                    <button type="submit" class="btn btn-primary btn-flat"><?php echo IconHelper::make('save') . t('app', 'Save changes'); ?></button>
                </div>
                <div class="clearfix"><!-- --></div>
            </div>
        </div>

        <?php if (!$server->getIsNewRecord()) { ?>
        <!-- modals -->
        <div class="modal modal-info fade" id="page-info" tabindex="-1" role="dialog">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <button type="button" class="close" data-dismiss="modal" aria-hidden="true">&times;</button>
                        <h4 class="modal-title"><?php echo IconHelper::make('info') . t('app', 'Info'); ?></h4>
                    </div>
                    <div class="modal-body">
                        <?php
                        $queryParams = urldecode(http_build_query([
                            'event'      => 'bounce',
                            'rule'       => '@bouncerule',
                            'type'       => '@bouncetype',
                            'campaign'   => sprintf('@%sCampaign-Uid', app_param('email.custom.header.prefix', '')),
                            'subscriber' => sprintf('@%sSubscriber-Uid', app_param('email.custom.header.prefix', '')),
                        ]));
                        echo t('servers', 'The url where this server expects to receive webhooks requests to process bounces is: {url}', [
                            '{url}' => sprintf('<strong>%s</strong>', apps()->getAppUrl('frontend', 'dswh/' . $server->server_id, true) . '/?' . $queryParams),
                        ]); ?><br /><br />
                        <?php
                        $queryParams = urldecode(http_build_query([
                            'event'      => 'complaint',
                            'campaign'   => sprintf('@%sCampaign-Uid', app_param('email.custom.header.prefix', '')),
                            'subscriber' => sprintf('@%sSubscriber-Uid', app_param('email.custom.header.prefix', '')),
                        ]));
                        echo t('servers', 'The url where this server expects to receive webhooks requests to process complaints is: {url}', [
                            '{url}' => sprintf('<strong>%s</strong>', apps()->getAppUrl('frontend', 'dswh/' . $server->server_id, true) . '/?' . $queryParams),
                        ]); ?><br /><br />
                        <?php
                        $queryParams = urldecode(http_build_query([
                            'event'      => 'unsubscribe',
                            'campaign'   => sprintf('@%sCampaign-Uid', app_param('email.custom.header.prefix', '')),
                            'subscriber' => sprintf('@%sSubscriber-Uid', app_param('email.custom.header.prefix', '')),
                        ]));
                        echo t('servers', 'The url where this server expects to receive webhooks requests to process unsubscribes is: {url}', [
                            '{url}' => sprintf('<strong>%s</strong>', apps()->getAppUrl('frontend', 'dswh/' . $server->server_id, true) . '/?' . $queryParams),
                        ]); ?><br /><br />
                        <?php echo t('servers', 'You can configure the above webhooks urls from: {url}', [
                            '{url}' => CHtml::link('https://email.dynect.net/index.php?Page=Integration', 'https://email.dynect.net/index.php?Page=Integration', ['target' => '_blank']),
                        ]); ?><br /><br />
                        <?php echo t('servers', 'Please note that you also have to register the following headers: {headers} from the above page, and you have to disable opens and click tracking from: {url}', [
                            '{headers}' => sprintf('<strong>%sCampaign-Uid, %sSubscriber-Uid</strong>', app_param('email.custom.header.prefix', ''), app_param('email.custom.header.prefix', '')),
                            '{url}'     => CHtml::link('https://email.dynect.net/index.php?Page=Users', 'https://email.dynect.net/index.php?Page=Users', ['target' => '_blank']),
                        ]); ?>
                    </div>
                </div>
            </div>
        </div>
        <?php } ?>

        <?php
        $controller->endWidget();
    }
    /**
     * This hook gives a chance to append content after the active form.
     * Please note that from inside the action callback you can access all the controller view variables
     * via {@CAttributeCollection $collection->controller->getData()}
     * @since 1.3.3.1
     */
    hooks()->doAction('after_active_form', new CAttributeCollection([
        'controller'      => $controller,
        'renderedForm'    => $collection->itemAt('renderForm'),
    ]));
}
/**
 * This hook gives a chance to append content after the view file default content.
 * Please note that from inside the action callback you can access all the controller view
 * variables via {@CAttributeCollection $collection->controller->getData()}
 * @since 1.3.3.1
 */
hooks()->doAction('after_view_file_content', new CAttributeCollection([
    'controller'        => $controller,
    'renderedContent'   => $viewCollection->itemAt('renderContent'),
]));
